// Copyright (c) 2024 Bastelbruder
// Webseite: www.bastelbruder.de
// Alle Rechte vorbehalten.

#include <Arduino.h>
#include <Wire.h>
#include <LiquidCrystal_I2C.h>
#include <DHT.h>

// ===================== Konfiguration =====================

// Pinbelegungen
#define DHTPIN 4            // Datenpin des DHT11
#define FAN_PIN 3           // Gate des MOSFET für den Lüfter
#define BUZZER_PIN 7        // Buzzer Pin

// DHT Sensortyp
#define DHTTYPE DHT11

// Zieltemperaturen mit integrierter Hysterese
const float TEMP_LEVELS[] = {26.0, 26.5, 27.5, 28.5}; // Temperaturstufen in Grad (inkl. Ausschaltstufe)
const int FAN_SPEEDS_PERCENT[] = {0, 60, 80, 100};    // Lüftergeschwindigkeiten in Prozent

const float BUZZER_TARGET_TEMP = 29.0;     // Zieltemperatur für den Buzzer in Grad Celsius (einstellbar)

// LCD Einstellungen
const uint8_t LCD_COLUMNS = 20;
const uint8_t LCD_ROWS = 4;
const uint8_t LCD_ADDRESS = 0x27; // I2C Adresse des LCD

/*
  - **PWM-Werte Bereich:** 0 bis 255 (0 = Lüfter aus, 255 = maximale Geschwindigkeit)
  - **Anpassung:** Du kannst die Prozentsätze anpassen, um die gewünschte Lüftergeschwindigkeit zu erreichen. Teste verschiedene Werte, um die optimale Leistung für deinen Lüfter und deine Anwendung zu finden.
  - **Feinabstimmung:** Falls dein Lüfter bei bestimmten PWM-Werten zu laut oder ineffizient arbeitet, passe die Werte entsprechend an.
*/

const size_t LEVEL_COUNT = sizeof(TEMP_LEVELS) / sizeof(TEMP_LEVELS[0]);

// ===========================================================

// Initialisierung der Objekte
DHT dht(DHTPIN, DHTTYPE);
LiquidCrystal_I2C lcd(LCD_ADDRESS, LCD_COLUMNS, LCD_ROWS);

// Variable zur Verfolgung des Lüfterstatus
bool fanIsOn = false;

// Variablen für die Zeitsteuerung
unsigned long previousMeasurementMillis = 0;
const unsigned long measurementInterval = 2000; // 2 Sekunden

// Variablen für die Alarmsteuerung
unsigned long previousAlarmMillis = 0;
const unsigned long alarmInterval = 500; // 500 Millisekunden für Piepen
bool buzzerState = false;

// ===================== Funktionen =====================

// Funktion zum Anzeigen der Soll- und Alarmtemperatur
void displayTemperatures() {
  lcd.setCursor(0, 2);
  lcd.print("Soll: ");
  lcd.print(TEMP_LEVELS[0], 1); // 26.0°C als Sollwert (0% Lüfter)
  lcd.write((byte)0xDF); // Gradzeichen
  lcd.print(" (");
  lcd.print(BUZZER_TARGET_TEMP, 1);
  lcd.write((byte)0xDF); // Gradzeichen
  lcd.print(")   "); // Zusätzliche Leerzeichen zur Bereinigung
}

// Funktion zur Umrechnung von Prozent auf PWM-Wert
int percentToPWM(int percent) {
  if (percent < 0) percent = 0;
  if (percent > 100) percent = 100;
  return map(percent, 0, 100, 0, 255);
}

// ===================== Setup =====================

void setup() {
  // Serielle Kommunikation starten (optional für Debugging)
  Serial.begin(9600);

  // Initialisierung des LCD
  lcd.init();           // Initialisiert das LCD
  lcd.backlight();      // LCD-Hintergrundbeleuchtung einschalten

  // Initialisierung der Pins
  pinMode(FAN_PIN, OUTPUT);
  pinMode(BUZZER_PIN, OUTPUT);

  // Startzustände
  analogWrite(FAN_PIN, 0); // Lüfter aus
  digitalWrite(BUZZER_PIN, LOW); // Buzzer aus

  // Initialisierung des DHT11 Sensors
  dht.begin();

  // Begrüßungsnachricht auf dem LCD
  lcd.setCursor(0, 0);
  lcd.print("Lueftersteuerung");
  lcd.setCursor(0, 1);
  lcd.print("Temp: --.- "); 
  lcd.write((byte)0xDF); // Gradzeichen
  lcd.setCursor(0, 2);
  displayTemperatures();
  lcd.setCursor(0, 3);
  lcd.print("Status: AUS       ");
}

// ===================== Loop =====================

void loop() {
  unsigned long currentMillis = millis();

  // Überprüfen, ob es Zeit für eine neue Messung ist
  if (currentMillis - previousMeasurementMillis >= measurementInterval) {
    previousMeasurementMillis = currentMillis;

    // Temperatur und Luftfeuchtigkeit lesen
    float temp = dht.readTemperature();
    float humidity = dht.readHumidity();

    // Überprüfen, ob die Lesung erfolgreich war
    if (isnan(temp) || isnan(humidity)) {
      Serial.println("Fehler beim Lesen des DHT11 Sensors!");
      lcd.setCursor(0, 1);
      lcd.print("Fehler beim Lesen  ");
      return;
    }

    // Debugging Ausgabe
    Serial.print("Aktuelle Temperatur: ");
    Serial.print(temp);
    Serial.println(" *C");

    // Anzeige aktualisieren
    lcd.setCursor(0, 1);
    lcd.print("Temp: ");
    lcd.print(temp, 1);
    lcd.write((byte)0xDF); // Gradzeichen
    lcd.print("       "); // Zusätzliche Leerzeichen zur Bereinigung

    // Hysterese-Logik für den Lüfter
    if (!fanIsOn && temp >= TEMP_LEVELS[1]) { // 26.5°C
      // Schalte den Lüfter ein
      fanIsOn = true;
      Serial.println("Lüfter eingeschaltet");
    } 
    else if (fanIsOn && temp <= TEMP_LEVELS[0]) { // 26.0°C
      // Schalte den Lüfter aus
      fanIsOn = false;
      analogWrite(FAN_PIN, 0);
      Serial.println("Lüfter ausgeschaltet");
    }

    // Bestimme die Lüftergeschwindigkeit basierend auf der Temperatur
    int fanSpeedPercent = 0;
    if (fanIsOn) {
      // Iteriere ab dem zweiten Element (Index 1), da Index 0 = 26.0°C (0%)
      for (size_t i = 1; i < LEVEL_COUNT; i++) {
        if (temp >= TEMP_LEVELS[i]) {
          fanSpeedPercent = FAN_SPEEDS_PERCENT[i];
        }
      }
      // Mindestgeschwindigkeit festlegen, falls keine höhere Stufe erreicht wurde
      if (fanSpeedPercent == 0) {
        fanSpeedPercent = FAN_SPEEDS_PERCENT[1]; // 60%
      }
      // Setze den PWM-Wert
      int pwmValue = percentToPWM(fanSpeedPercent);
      analogWrite(FAN_PIN, pwmValue);
    } 
    else {
      // Lüfter ausschalten
      analogWrite(FAN_PIN, 0);
      fanSpeedPercent = 0;
    }

    // Statusanzeige aktualisieren
    lcd.setCursor(0, 3);
    if (fanSpeedPercent > 0) {
      lcd.print("Luefter: ");
      lcd.print(fanSpeedPercent);
      lcd.print("%  "); // Zwei Leerzeichen nach %
    } else {
      lcd.print("Luefter: AUS       ");
    }

    // Buzzer Steuerung: Alarm aktivieren oder deaktivieren
    if (temp > BUZZER_TARGET_TEMP) { // Aktiviert den Buzzer nur, wenn temp > BUZZER_TARGET_TEMP
      // Alarm-Logik aktivieren
      // Keine Aktion hier, da die Alarm-Logik unten läuft
    } else {
      // Alarm deaktivieren
      digitalWrite(BUZZER_PIN, LOW);
      buzzerState = false; // Reset des Buzzer-Zustands
      lcd.setCursor(14, 3);
      lcd.print("      "); // Löscht den Alarmtext mit Leerzeichen
    }
  }

  // Zusätzliche Alarm-Logik außerhalb des Messintervalls
  // Diese Logik läuft kontinuierlich und steuert den Buzzer
  float temp = dht.readTemperature(); // Aktuelle Temperatur erneut lesen

  if (temp > BUZZER_TARGET_TEMP) {
    unsigned long currentMillis = millis();
    if (currentMillis - previousAlarmMillis >= alarmInterval) {
      previousAlarmMillis = currentMillis;
      buzzerState = !buzzerState;
      digitalWrite(BUZZER_PIN, buzzerState ? HIGH : LOW);
    }
    // Anzeige des Alarms auf dem LCD
    lcd.setCursor(14, 3);
    lcd.print("ALARM!");
  } else {
    // Buzzer ausschalten, wenn kein Alarm
    digitalWrite(BUZZER_PIN, LOW);
    buzzerState = false;
    lcd.setCursor(14, 3);
    lcd.print("      "); // Löscht den Alarmtext mit Leerzeichen
  }
}
